<?php
/**
 * Momo Standard Payment Gateway.
 *
 * Provides a Momo Standard Payment Gateway.
 *
 * @class       WC_Gateway_Momo
 * @extends     WC_Payment_Gateway
 * @version     1.0.0
 * @package     WooCommerce\Classes\Payment
 */

use Automattic\Jetpack\Constants;

if (!defined('ABSPATH')) {
    exit;
}

/**
 * WC_Gateway_Momo Class.
 */
class WC_Gateway_Momo extends WC_Payment_Gateway
{

    /**
     * Whether or not logging is enabled
     *
     * @var bool
     */
    public static $log_enabled = false;

    /**
     * Logger instance
     *
     * @var WC_Logger
     */
    public static $log = false;

    /**
     * Constructor for the gateway.
     */
    public function __construct()
    {
        $this->id                = 'momo';
        $this->has_fields        = false;
        $this->order_button_text = __('Proceed to MoMo', 'woocommerce');
        $this->method_title      = __('MoMo Checkout', 'woocommerce');
        /* translators: %s: Link to WC system status page */
        $this->method_description = __('Thanh toán trực tiếp qua Momo.', 'woocommerce');
        $this->supports           = array(
            'products',
            'refunds',
        );

        // Load the settings.
        $this->init_form_fields();
        $this->init_settings();

        // Define user set variables.
        $this->title       = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->debug       = 'yes' === $this->get_option('debug', 'no');

        self::$log_enabled = $this->debug;

        add_action('woocommerce_update_options_payment_gateways_'.$this->id, array($this, 'process_admin_options'));
        add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

        // Add payment gateways
        add_filter('woocommerce_payment_gateways', array($this, 'payment_gateways'));

        if (!$this->is_valid_for_use()) {
            $this->enabled = 'no';
        }

        if ('yes' === $this->enabled) {
            add_filter('woocommerce_thankyou_order_received_text', array($this, 'order_received_text'), 10, 2);
        }
    }

    public function payment_gateways($load_gateways)
    {
        $load_gateways[] = 'WC_Gateway_Momo';

        return $load_gateways;
    }

    /**
     * Return whether or not this gateway still requires setup to function.
     *
     * When this gateway is toggled on via AJAX, if this returns true a
     * redirect will occur to the settings page instead.
     *
     * @return bool
     * @since 3.4.0
     */
    public function needs_setup()
    {
        return !is_email($this->email);
    }

    /**
     * Logging method.
     *
     * @param  string  $message  Log message.
     * @param  string  $level  Optional. Default 'info'. Possible values:
     *                      emergency|alert|critical|error|warning|notice|info|debug.
     */
    public static function log($message, $level = 'info')
    {
        if (self::$log_enabled) {
            if (empty(self::$log)) {
                self::$log = wc_get_logger();
            }
            self::$log->log($level, $message, array('source' => 'momo'));
        }
    }

    /**
     * Processes and saves options.
     * If there is an error thrown, will continue to save and validate fields, but will leave the erroring field out.
     *
     * @return bool was anything saved?
     */
    public function process_admin_options()
    {
        $saved = parent::process_admin_options();

        // Maybe clear logs.
        if ('yes' !== $this->get_option('debug', 'no')) {
            if (empty(self::$log)) {
                self::$log = wc_get_logger();
            }
            self::$log->clear('momo');
        }

        return $saved;
    }

    /**
     * Get gateway icon.
     *
     * @return string
     */
    public function get_icon()
    {
        // We need a base country for the link to work, bail if in the unlikely event no country is set.
        $base_country = WC()->countries->get_base_country();
        if (empty($base_country)) {
            return '';
        }
        $icon_html = '';
        $icon      = (array) $this->get_icon_image();

        foreach ($icon as $i) {
            $icon_html .= '<img src="'.esc_attr($i).'" alt="'.esc_attr__('MoMo acceptance mark', 'woocommerce').'" />';
        }

        $icon_html .= sprintf('<a href="%1$s" class="about_momo" onclick="javascript:window.open(\'%1$s\',\'WIMomo\',\'toolbar=no, location=no, directories=no, status=no, menubar=no, scrollbars=yes, resizable=yes, width=1060, height=700\'); return false;">'.esc_attr__('What is MoMo?', 'woocommerce').'</a>', esc_url($this->get_icon_url()));

        return apply_filters('woocommerce_gateway_icon', $icon_html, $this->id);
    }

    /**
     * Get the link for an icon based on country.
     *
     * @return string
     */
    protected function get_icon_url()
    {
        return 'https://developers.momo.vn/images/logo.png';
    }

    /**
     * Get MoMo images for a country.
     *
     * @return array of image URLs
     */
    protected function get_icon_image()
    {
        $icon = WC_HTTPS::force_https_url('https://developers.momo.vn/images/logo.png');

        return apply_filters('woocommerce_momo_icon', $icon);
    }

    /**
     * Check if this gateway is available in the user's country based on currency.
     *
     * @return bool
     */
    public function is_valid_for_use()
    {
        return in_array(
            get_woocommerce_currency(),
            apply_filters(
                'woocommerce_momo_supported_currencies',
                array(
                    'VND',
                    'USD',
                )
            ),
            true
        );
    }

    /**
     * Admin Panel Options.
     * - Options for bits like 'title' and availability on a country-by-country basis.
     *
     * @since 1.0.0
     */
    public function admin_options()
    {
        if ($this->is_valid_for_use()) {
            parent::admin_options();
        } else {
            ?>
            <div class="inline error">
                <p>
                    <strong><?php esc_html_e('Gateway disabled', 'woocommerce'); ?></strong>: <?php esc_html_e('MoMo Standard does not support your store currency.', 'woocommerce'); ?>
                </p>
            </div>
            <?php
        }
    }

    /**
     * Initialise Gateway Settings Form Fields.
     */
    public function init_form_fields()
    {
        $this->form_fields = include __DIR__.'/includes/settings-gateway.php';
    }

    /**
     * Load admin scripts.
     *
     * @since 3.3.0
     */
    public function admin_scripts()
    {
        $screen    = get_current_screen();
        $screen_id = $screen ? $screen->id : '';

        if ('woocommerce_page_wc-settings' !== $screen_id) {
            return;
        }

        $suffix  = Constants::is_true('SCRIPT_DEBUG') ? '' : '.min';
        $version = Constants::get_constant('WC_VERSION');

        wp_enqueue_script('woocommerce_momo_admin', WC()->plugin_url().'/includes/gateways/momo/assets/js/admin'.$suffix.'.js', array(), $version, true);
    }

    /**
     * Custom MoMo order received text.
     *
     * @param  string  $text  Default text.
     * @param  WC_Order  $order  Order data.
     *
     * @return string
     * @since 3.9.0
     */
    public function order_received_text($text, $order)
    {
        if ($order && $this->id === $order->get_payment_method()) {
            return esc_html__('Thank you for your payment. Your transaction has been completed, and a receipt for your purchase has been emailed to you. Log into your MoMo account to view transaction details.', 'woocommerce');
        }

        return $text;
    }
}

new WC_Gateway_Momo();
